#include <iostream>
#include <fstream>
#include <vector>
#include <algorithm>
#include <chrono>
#include <random>

using namespace std;

typedef vector<int>::iterator iter;

const int max_number = 10000;

ostream &operator <<(ostream &to, const vector<int> &v) {
	to << "[";

	if (v.size() > 0)
		to << v[0];

	for (size_t i = 1; i < v.size(); i++)
		to << ", " << v[i];

	to << "]";
	return to;
}

bool is_sorted(const vector<int> &v) {
	for (size_t i = 1; i < v.size(); i++) {
		if (v[i - 1] > v[i])
			return false;
	}
	return true;
}

void count_sort(vector<int> &data) {
	vector<int> count(max_number, 0);
	for (int x : data)
		count[x]++;

	size_t pos = 0;
	for (int i = 0; i < int(max_number); i++)
		for (int c = 0; c < count[i]; c++)
			data[pos++] = i;
}
// end COUNT

const size_t num_buckets = 100;
void bucket_sort(vector<int> &data) {
	vector<vector<int>> buckets(num_buckets);
	const int divide = (max_number / num_buckets);
	for (int x : data)
		buckets[x / divide].push_back(x);

	size_t pos = 0;
	for (vector<int> &x : buckets) {
		sort(x.begin(), x.end());
		copy(x.begin(), x.end(), data.begin() + pos);
		pos += x.size();
	}
}
// end BUCKET

// begin BUCKET_DIGIT:
void bucket_sort_digit(vector<int> &data, int div) {
	vector<vector<int>> buckets(10);

	for (int x : data)
		buckets[(x / div) % 10].push_back(x);

	size_t pos = 0;
	for (vector<int> &x : buckets) {
		copy(x.begin(), x.end(), data.begin() + pos);
		pos += x.size();
	}
}
// end BUCKET_DIGIT

void radix_sort(vector<int> &data) {
	for (int divide = 1;
		 divide < 10 * max_number;
		 divide *= 10) {

		bucket_sort_digit(data, divide);
	}
}
// end RADIX

void std_sort(vector<int> &data) {
	std::sort(data.begin(), data.end());
}

template <typename T>
void sort(vector<int> data, T fn) {
	fn(data);
	cout << data << endl;
	if (!is_sorted(data))
		cout << "ERROR! Not sorted!" << endl;
}

template <typename T>
size_t time(vector<int> data, T fn) {
	auto a = chrono::high_resolution_clock::now();
	fn(data);
	auto b = chrono::high_resolution_clock::now();
	if (!is_sorted(data))
		cerr << "ERROR! Not sorted!" << endl;
	return chrono::duration_cast<chrono::milliseconds>(b - a).count();
}

void time_fns(ostream &to, const vector<int> &) {
	to << endl;
}

template <typename First, typename... Rest>
void time_fns(ostream &to, const vector<int> &data, First first, Rest...fn) {
	to << "\t" << time(data, first);
	time_fns(to, data, fn...);
}

template <typename... T>
void time(const string &file, size_t elements, size_t step, T...fn) {
	mt19937 random;
	vector<int> data(elements, 0);
	uniform_int_distribution<int> dist(0, max_number - 1);
	generate(data.begin(), data.end(), [&dist, &random]() { return dist(random); });

	ofstream out(file);
	vector<int> to_sort;
	for (size_t i = 0; i <= data.size(); i += step) {
		cerr << i << " of " << data.size() << "..." << endl;

		while (to_sort.size() < i)
			to_sort.push_back(data[to_sort.size()]);

		out << i;
		time_fns(out, to_sort, fn...);
	}
}

void sanity_check() {
	// Simple test to check that the sorting is correct.
	vector<int> data{5, 1000, 800, 7, 2};
	cout << data << endl;

	sort(data, count_sort);
	sort(data, bucket_sort);
	sort(data, radix_sort);
}

int main() {
	sanity_check();

	cerr << "Comparison between all..." << endl;
	time("sort.txt", 2000000, 20000, count_sort, bucket_sort, radix_sort, std_sort);

	return 0;
}
