#include <iostream>
#include <iomanip>
#include <algorithm>
#include <string>
#include <sstream>
#include <vector>
#include <functional>

#include "dump.h"
#include "column.h"
#include "columns.h"

using std::min;
using std::max;
using std::cout;
using std::endl;

using std::string;
using std::vector;
using std::ostream;

using Row = vector<string>;
using Table = vector<Row>;

Columns numeric() {
	vector<Column> columns;
	columns.push_back(columns::offset());
	columns.push_back(columns::bytes());
	if (sizeof(int) != sizeof(size_t)) {
		columns.push_back(columns::low_int());
		columns.push_back(columns::high_int());
	}
	columns.push_back(columns::word());
	return columns;
}

Columns pointer() {
	vector<Column> columns;
	columns.push_back(columns::offset());
	columns.push_back(columns::pointer());
	columns.push_back(columns::symbol());
	return columns;
}



static Row build_row(RowInfo info, const vector<Column> &columns) {
	Row result;
	for (size_t i = 0; i < columns.size(); i++) {
		std::ostringstream out;
		columns[i].generator(out, info);
		result.push_back(out.str());
	}
	return result;
}

static Table build_table(const void *memory, size_t size, size_t adjust_offset, const vector<Column> &columns) {
	Table table;

	{
		Row header;
		for (size_t i = 0; i < columns.size(); i++)
			header.push_back(columns[i].header);
		table.push_back(header);
	}

	const byte *data = reinterpret_cast<const byte *>(memory);
	for (size_t offset = 0; offset < size; offset += sizeof(void *)) {
		RowInfo row = { offset + adjust_offset, data + offset, min(size - offset, sizeof(void *)) };
		table.push_back(build_row(row, columns));
	}

	return table;
}

static void print_table(const Table &table, const vector<Column> &columns) {
	vector<int> widths(table[0].size(), 0);
	for (const Row &row : table) {
		for (size_t col = 0; col < row.size(); col++) {
			widths[col] = max(widths[col], int(row[col].size()));
		}
	}

	std::ios_base::fmtflags old = cout.flags();

	for (size_t row = 0; row < table.size(); row++) {
		for (size_t col = 0; col < table[row].size(); col++) {
			if (col > 0)
				cout << " | ";

			if (row == 0) {
				cout << std::left;
			} else {
				cout << *columns[col].align;
			}

			cout << std::setw(widths[col]) << table[row][col];
		}
		cout << endl;

		if (row == 0) {
			for (size_t col = 0; col < widths.size(); col++) {
				if (col > 0)
					cout << "-|-";
				cout << std::setfill('-') << std::setw(widths[col]) << "-";
			}
			cout << endl;
			cout << std::setfill(' ');
		}
	}

	cout.flags(old);
}

void dump_memory(const void *memory, size_t size, size_t adjust_offset, const Columns &columns) {
	Table table = build_table(memory, size, adjust_offset, columns);

	cout << "Dumping " << size << " bytes from " << memory << endl;
	print_table(table, columns);
	cout << endl;
}

void dump_memory(const void *memory, size_t size, const Columns &columns) {
	dump_memory(memory, size, 0, columns);
}

void dump_memory(const void *start, void *end, const Columns &columns) {
	dump_memory(start, (byte *)end - (byte *)start, columns);
}
