#include "columns.h"
#include "syminfo.h"

#include <iomanip>

using std::ostream;

namespace columns {

	static void gen_offset(ostream &to, RowInfo info) {
		to << std::hex << std::setfill('0') << std::setw(4) << (info.offset & 0xFFFF);
	}

	Column offset() {
		return Column("Off.", std::right, gen_offset);
	}

	static void gen_bytes(ostream &out, RowInfo info) {
		out << std::hex << std::setfill('0');
		for (size_t i = 0; i < info.size; i++) {
			if (i > 0)
				out << " ";
			out << std::setw(2) << (unsigned)info.data[i];
		}
	}

	Column bytes() {
		return Column("Bytes", std::left, gen_bytes);
	}

	static void gen_low_int(ostream &out, RowInfo info) {
		if (info.size >= sizeof(int))
			out << *reinterpret_cast<const int *>(info.data);
		else
			out << "-";
	}

	static void gen_high_int(ostream &out, RowInfo info) {
		if (info.size >= 2*sizeof(int))
			out << *reinterpret_cast<const int *>(info.data + sizeof(int));
		else
			out << "-";
	}

	Column low_int() {
		return Column("Int", std::right, gen_low_int);
	}

	Column high_int() {
		return Column("Int", std::right, gen_high_int);
	}

	static void gen_word(ostream &out, RowInfo info) {
		if (info.size >= sizeof(size_t))
			out << *reinterpret_cast<const size_t *>(info.data);
		else
			out << "-";
	}

	Column word() {
		return Column("Long", std::right, gen_word);
	}

	static void gen_pointer(ostream &out, RowInfo info) {
		if (info.size >= sizeof(size_t))
			out << std::hex << std::setfill('0') << std::setw(sizeof(size_t)*2)
				<< *reinterpret_cast<const size_t *>(info.data);
		else
			out << "-";
	}

	Column pointer() {
		return Column("Pointer", std::right, gen_pointer);
	}

	static void gen_symbol(ostream &out, RowInfo info) {
		if (info.size >= sizeof(size_t))
			out << syminfo(*reinterpret_cast<void *const*>(info.data));
		else
			out << "-";
	}

	Column symbol() {
		return Column("Symbol", std::left, gen_symbol);
	}

}
